"""Rate limiter en memoria para proteger endpoints criticos."""
from collections import defaultdict
from datetime import datetime, timedelta
import threading

# Config
MAX_SELLS_PER_MINUTE = 10

# Registro de timestamps por agente: {"agent_id": [datetime, datetime, ...]}
_sell_log: dict[str, list[datetime]] = defaultdict(list)
_lock = threading.Lock()


def check_sell_rate(agent_id: str) -> bool:
    """
    Verifica si el agente puede hacer otra venta.
    Retorna True si esta dentro del limite, False si excedio.
    """
    if not agent_id:
        # Sin agent_id no aplicamos rate limit (sell busca el primero disponible)
        return True

    now = datetime.now()
    cutoff = now - timedelta(minutes=1)

    with _lock:
        # Limpiar entradas viejas
        _sell_log[agent_id] = [t for t in _sell_log[agent_id] if t > cutoff]

        if len(_sell_log[agent_id]) >= MAX_SELLS_PER_MINUTE:
            return False

        # Registrar esta venta
        _sell_log[agent_id].append(now)
        return True


def get_sell_count(agent_id: str) -> int:
    """Retorna cuantas ventas ha hecho el agente en el ultimo minuto."""
    now = datetime.now()
    cutoff = now - timedelta(minutes=1)

    with _lock:
        _sell_log[agent_id] = [t for t in _sell_log[agent_id] if t > cutoff]
        return len(_sell_log[agent_id])
